<?php
/*
Plugin Name: Custom Shipping Options
Description: Fetches dynamic shipping Options using OAuth and external API.
Version: 1.2
Author: OES
*/


if (!defined('ABSPATH')) exit;

require_once plugin_dir_path(__FILE__) . 'includes/class-api-handler.php';
require_once plugin_dir_path(__FILE__) . 'includes/class-frontend-display.php';

require_once plugin_dir_path(__FILE__) . 'includes/admin-settings.php';

// Redirect after activation
register_activation_hook(__FILE__, 'oes_plugin_activate');
function oes_plugin_activate() {
    add_option('oes_plugin_do_redirect', true);
}

// Redirect to settings page
add_action('admin_init', 'oes_plugin_redirect_to_settings');
function oes_plugin_redirect_to_settings() {
    if (get_option('oes_plugin_do_redirect', false)) {
        delete_option('oes_plugin_do_redirect');
        wp_safe_redirect(admin_url('admin.php?page=oes-plugin-settings'));
        exit;
    }
}

//  function convert_currency($amount, $from, $to) {
//     //$response = wp_remote_get("https://api.exchangerate.host/convert?from={$from}&to={$to}&amount={$amount}");
    
//     $response = wp_remote_get("https://api.frankfurter.app/latest?from={$from}&to={$to}&amount={$amount}");

//     if (is_wp_error($response)) return $amount;

//     $data = json_decode(wp_remote_retrieve_body($response), true);
//     error_log('api.exchangerate.host/convert amount :'. json_encode($data));
//     error_log('from amount :('. $from.') : '. json_encode($amount));
//     error_log('to amount :('. $to.') : '. json_encode(round($amount * $data['result'], 2)));

//     if (!isset($data['result'])) return $amount;

//     return round($amount * $data['result'], 2);
// }

function convert_currency($amount, $from_currency, $to_currency) {
    // Sanitize and prepare API URL
    $amount = floatval($amount);
    $from_currency = strtoupper(sanitize_text_field($from_currency));
    $to_currency = strtoupper(sanitize_text_field($to_currency));

    if ($from_currency === $to_currency) {
        return round($amount, 2); // No conversion needed
    }

    $url = "https://api.frankfurter.app/latest?amount={$amount}&from={$from_currency}&to={$to_currency}";

    // Make the HTTP request
    $response = wp_remote_get($url);

    if (is_wp_error($response)) {
        //error_log('Currency conversion error: ' . $response->get_error_message());
        return $amount;
    }

    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);

    if (!isset($data['rates'][$to_currency])) {
        //error_log('Currency conversion failed: Invalid API response');
        return $amount;
    }

    return round(floatval($data['rates'][$to_currency]), 2);
}

// Register a custom shipping method to show only our API-based quotes
add_action('woocommerce_shipping_init', 'oes_custom_shipping_method_init');
function oes_custom_shipping_method_init() {
    class WC_OES_Custom_Shipping_Method extends WC_Shipping_Method {
        public function __construct() {
            $this->id                 = 'oes_custom_shipping';
            $this->method_title       = 'OES Shipping';
            $this->method_description = 'Shipping method using OES custom API quotes';
            $this->enabled            = 'yes';
            $this->title              = 'Custom API Shipping';
            $this->init();
        }

        public function init() {
            $this->init_form_fields();
            $this->init_settings();
            add_action('woocommerce_update_options_shipping_' . $this->id, [ $this, 'process_admin_options' ]);
        }

        
        

        public function calculate_shipping($package = []) {
            $quotes = WC()->session->get('custom_quote');

            if (!is_array($quotes)) {
                error_log('No custom quotes in session');
                return;
            }
            
            foreach ($quotes['data'] as $index => $quote) {
                $rate_id = $this->id . ':' . sanitize_title($quote['provider']['name'] . '-' . $index);
                $delivery_text = ($quote['delivery_time'] === 0) ? 'Today' : $quote['delivery_time'] . ' days';

                // Prepare carrier logo if available
                $logo_html = '';
                if (!empty($quote['provider']['image'])) {
                    $logo_html = '<img src="' . $quote['provider']['image'] . '" style="width:100px; height: auto; margin-right: 8px; vertical-align: middle;" />';
                }
                $logo_url = $quote['provider']['image'];
                
                $carrier_label = esc_html($quote['provider']['name'] .' '. $quote['description'] .' '. $delivery_text);
                $label = $logo_html . '<span style="vertical-align: middle;">' . $carrier_label . '</span>';

                $store_currency = get_woocommerce_currency();
                $quote_currency = $quote['currency']['name'] ?? 'USD';
                $freight = floatval($quote['freight']);
                $cost = ($quote_currency !== $store_currency)
                    ? convert_currency($freight, $quote_currency, $store_currency)
                    : $freight;

                $rate = [
                    'id'    => $rate_id,
                    //'label' => $quote['provider']['name'] .' - '.$delivery_text,
                    'label' => sprintf(
                        '<img src="%s" style="height:20px;vertical-align:middle;margin-right:8px;"> <span style="vertical-align:middle;">%s</span>',
                        $logo_url,
                        $carrier_label
                    ),
                    'cost'  => floatval($cost),
                    'meta_data' => [
                        'delivery_time' => $delivery_text ?? '',
                        'oes_order_id' => $quote['quotation_id'],
                        'oes_option_id' => $quote['id'],
                        'quote_info'    => $quote,
                    ],
                ];
                $this->add_rate($rate);
            }
        }
    }
}

add_filter('woocommerce_shipping_methods', 'add_oes_custom_shipping_method');
function add_oes_custom_shipping_method($methods) {
    $methods['oes_custom_shipping'] = 'WC_OES_Custom_Shipping_Method';
    return $methods;
}

// Optional: Disable all other shipping methods (leave only our custom method)
add_filter('woocommerce_package_rates', 'filter_shipping_methods_to_only_custom', 100, 2);
function filter_shipping_methods_to_only_custom($rates, $package) {
    foreach ($rates as $key => $rate) {
        if (strpos($key, 'oes_custom_shipping') !== 0) {
            unset($rates[$key]);
        }
    }
    return $rates;
}

// Hook into cart/checkout to fetch quotes early
add_action('woocommerce_cart_calculate_fees', 'trigger_custom_shipping_quote_fetch', 10);
function trigger_custom_shipping_quote_fetch() {
    
    if (!is_admin() && is_checkout()) {
        if (method_exists('Custom_API_Handler', 'oes_fetch_token')) {
            Custom_API_Handler::oes_fetch_token();
        }
        if (method_exists('Custom_API_Handler', 'oes_get_quote')) {
            Custom_API_Handler::oes_get_quote();
        }
    }
}

add_action('woocommerce_thankyou', ['Custom_API_Handler', 'oes_create_shipment_after_payment']);


add_action('woocommerce_thankyou', function($order_id) {
    $tracking = get_post_meta($order_id, '_oes_tracking_number', true);
    if ($tracking) {
        echo "<p><strong>Tracking Number:</strong> {$tracking}</p>";
    }
});

add_action('wp_footer', 'oes_inject_carrier_logo_script');
function oes_inject_carrier_logo_script() {
    if (!is_checkout()) return;

    $quotes = WC()->session->get('custom_quote');
    if (!is_array($quotes) || empty($quotes['data'])) return;

    $carriers = [];
    foreach ($quotes['data'] as $quote) {
        $delivery_text = ($quote['delivery_time'] === 0) ? 'Today' : $quote['delivery_time'] . ' days';
        $name  = $quote['provider']['name'] .' '.$quote['description'].' '.$delivery_text; // Simplify to just the provider name
        $image = $quote['provider']['image'] ?? '';
        if ($name && $image) {
            $carriers[$name] = esc_url($image);
        }
    }
?>
    
    <style>
        #block-shipping-filter button {
            margin-right: 8px;
            padding: 5px 12px;
            border: 1px solid #ccc;
            background: #f7f7f7;
            cursor: pointer;
        }
        #block-shipping-filter{
            text-align:right !important;
        }
        

        #block-shipping-filter button:hover {
            background: #e0e0e0;
        }
       
    </style>

    
    <script>
    
        const carriers = <?php echo json_encode($carriers); ?>;

        function injectCarrierLogos() {
            const labels = document.querySelectorAll('label[for^="radio-control-0-oes_custom_shipping"]');

            labels.forEach(function (label) {
                const labelText = label.textContent.trim();

                // Skip if already injected
                if (label.querySelector('img')) return;

                for (const keyword in carriers) {
                    if (labelText.includes(keyword)) {
                        const logoUrl = carriers[keyword];

                        // Find the label text span to inject logo before it
                        const labelGroup = label.querySelector('.wc-block-components-radio-control__label-group');
                        const mainLabel = label.querySelector('.wc-block-components-radio-control__label');

                        if (labelGroup && mainLabel) {
                            const img = document.createElement('img');
                            img.src = logoUrl;
                            img.style.height = '30px';
                            img.style.verticalAlign = 'middle';
                            img.style.marginRight = '8px';

                            mainLabel.prepend(img); // Inject logo before text
                        }
                        break;
                    }
                }
            });
        }

        function FilterWC() {
    const heading = document.querySelector('.wc-block-checkout__shipping-option');
    if (!heading) return;

    const filterDiv = document.createElement('div');
    filterDiv.id = 'block-shipping-filter';
    filterDiv.innerHTML = `
        <div style="margin-bottom: 15px;">
            
            <button type="button" class="shipping-filter-btn" data-filter="fastest">Fastest Delivery</button>
            <button type="button" class="shipping-filter-btn" data-filter="cheapest">Lowest Price</button>
        </div>
    `;

    const title = heading.querySelector('.wc-block-components-title');
    if (title && title.parentNode) {
        title.parentNode.insertBefore(filterDiv, title.nextSibling);
    }

    document.querySelectorAll('.shipping-filter-btn').forEach(button => {
        button.addEventListener('click', () => {
            const methodList = heading.querySelectorAll('.wc-block-components-radio-control__option');
            if (!methodList.length) return;

            const methods = Array.from(methodList).map(el => {
                const label = el.textContent.toLowerCase();
                const priceMatch = label.match(/\$([\d.,]+)/);

                let days = 999;
                if (label.includes('today')) {
                    days = 0;
                } else {
                    const match = label.match(/(\d+)\s*days?/);
                    if (match) days = parseInt(match[1]);
                }

                const price = priceMatch ? parseFloat(priceMatch[1].replace(/,/g, '')) : Infinity;
                return { el, price, days };
            });

            if (button.dataset.filter === 'fastest') {
                methods.sort((a, b) => a.days - b.days);
            } else if (button.dataset.filter === 'cheapest') {
                methods.sort((a, b) => a.price - b.price);
            }

            const container = methodList[0].parentNode;
            methods.forEach(method => container.appendChild(method.el));

            // Select first
            // const radio = methods[0].el.querySelector('input[type="radio"]');
            // if (radio) {
            //     radio.checked = true;
            //     radio.click();
            //     radio.dispatchEvent(new Event('change', { bubbles: true }));
            // }

            const label = methods[0].el.querySelector('label');
            if (label) {
                label.click();
            }
        });
    });
}

        // Initial run after slight delay (for block UI load)
        document.addEventListener('DOMContentLoaded', function () {
        setTimeout(injectCarrierLogos, 1000);
        setTimeout(FilterWC, 1000);
    });
        // Re-run after checkout reloads shipping
        document.body.addEventListener('updated_checkout', injectCarrierLogos);
        document.body.addEventListener('updated_shipping_method', injectCarrierLogos);

    </script>
    <?php
}




add_filter('woocommerce_add_to_cart_validation', 'force_single_product_cart_with_notice', 99, 2);
function force_single_product_cart_with_notice($passed, $product_id) {
    if (!WC()->cart->is_empty()) {
        wc_add_notice(__('Only one product can be purchased at a time. Your cart has been updated.'), 'notice');
        WC()->cart->empty_cart();
    }
    return $passed;
}
add_filter('woocommerce_cart_needs_shipping', 'disable_shipping_on_cart_page');
function disable_shipping_on_cart_page($needs_shipping) {
    if (is_cart()) {
        return false;
    }
    return $needs_shipping;
}

add_action('woocommerce_admin_order_data_after_order_details', 'render_oes_shipping_block_and_move', 10, 1);

function render_oes_shipping_block_and_move($order) {
    $tracking = $order->get_meta('oes_tracking_number');
    $label    = $order->get_meta('oes_label');
    $invoice  = $order->get_meta('oes_invoice');
    $oes_provider_logo = $order->get_meta('oes_provider_logo');

    if (!$tracking && !$label && !$invoice) {
        return;
    }

    // Output a uniquely identifiable wrapper div
    ?>
    <div id="oes-shipping-info-temp" style="display: none;">
        <div class="panel woocommerce-panel" style="margin-top:20px;padding: 20px;
    background: #fff;
    border: 1px solid #d3cfcf;margin-bottom:20px">
            <h3>OES Shipping Details</h3>
            <ul>
            <?php if ($oes_provider_logo): ?>
                    <li style="margin:20px 0px;"><img src="<?= esc_html($oes_provider_logo); ?>" alt="Shipping Service Provider logo" width="200" ></li>
                <?php endif; ?>
                <?php if ($tracking): ?>
                    <li style="margin:20px 0px;"><strong>Tracking Number:</strong> <?= esc_html($tracking); ?></li>
                <?php endif; ?>
                <?php if ($label): ?>
                    <li style="margin:20px 0px;"><strong>Label:</strong> <a style="padding-left:20px;" href="<?= esc_url($label); ?>" target="_blank">View Label</a></li>
                <?php endif; ?>
                <?php if ($invoice): ?>
                    <li style="margin:20px 0px;"><strong>Invoice:</strong> <a style="padding-left:20px;" href="<?= esc_url($invoice); ?>" target="_blank">View Invoice</a></li>
                <?php endif; ?>
            </ul>
        </div>
    </div>

    <script type="text/javascript">
        document.addEventListener('DOMContentLoaded', function () {
            const temp = document.getElementById('oes-shipping-info-temp');
            const itemsPanel = document.querySelector('#woocommerce-order-items');

            if (temp && itemsPanel) {
                const clone = temp.querySelector('.panel').cloneNode(true);
                itemsPanel.parentNode.insertBefore(clone, itemsPanel);
            }
        });
    </script>
    <?php
}





